<?php
/**
 * Widget: Google Maps
 *
 * @package thelandscaper-toolkit
 */

if ( ! class_exists( 'QT_Google_Map' ) ) {
	class QT_Google_Map extends WP_Widget {

		/**
		 * Register widget with WordPress.
		 */
		public function __construct() {
			parent::__construct(
		 		false,
				esc_html__( 'QT: Google Maps (API & Billed)', 'the-landscaper-wp' ),
				array( 
					'description' => esc_html__( 'Display API embedded map (Google billing account & API key required)', 'the-landscaper-wp' ),
					'classname'   => 'widget-google-map',
				)
			);
		}

		/**
		 * Custom Map Styles
		 * @see snazzymaps.com
		 */
		private $map_styles = array(
			'Default'          => '[]',
        	'Paper'            => '[{"featureType":"administrative","elementType":"all","stylers":[{"visibility":"off"}]},{"featureType":"landscape","elementType":"all","stylers":[{"visibility":"simplified"},{"hue":"#0066ff"},{"saturation":74},{"lightness":100}]},{"featureType":"poi","elementType":"all","stylers":[{"visibility":"simplified"}]},{"featureType":"road","elementType":"all","stylers":[{"visibility":"simplified"}]},{"featureType":"road.highway","elementType":"all","stylers":[{"visibility":"off"},{"weight":0.6},{"saturation":-85},{"lightness":61}]},{"featureType":"road.highway","elementType":"geometry","stylers":[{"visibility":"on"}]},{"featureType":"road.arterial","elementType":"all","stylers":[{"visibility":"off"}]},{"featureType":"road.local","elementType":"all","stylers":[{"visibility":"on"}]},{"featureType":"transit","elementType":"all","stylers":[{"visibility":"simplified"}]},{"featureType":"water","elementType":"all","stylers":[{"visibility":"simplified"},{"color":"#5f94ff"},{"lightness":26},{"gamma":5.86}]}]',
        	'Paper with Roads' => '[{"featureType":"administrative","elementType":"all","stylers":[{"visibility":"off"}]},{"featureType":"landscape","elementType":"all","stylers":[{"visibility":"simplified"},{"hue":"#0066ff"},{"saturation":74},{"lightness":100}]},{"featureType":"poi","elementType":"all","stylers":[{"visibility":"simplified"}]},{"featureType":"road","elementType":"all","stylers":[{"visibility":"simplified"}]},{"featureType":"road.highway","elementType":"all","stylers":[{"visibility":"off"},{"weight":0.6},{"saturation":-85},{"lightness":61}]},{"featureType":"road.highway","elementType":"geometry","stylers":[{"visibility":"on"}]},{"featureType":"road.arterial","elementType":"all","stylers":[{"visibility":"on"}]},{"featureType":"road.local","elementType":"all","stylers":[{"visibility":"on"}]},{"featureType":"transit","elementType":"all","stylers":[{"visibility":"simplified"}]},{"featureType":"water","elementType":"all","stylers":[{"visibility":"simplified"},{"color":"#5f94ff"},{"lightness":26},{"gamma":5.86}]}]',
        	'Canary'           => '[{"featureType":"all","elementType":"all","stylers":[{"hue":"#ffbb00"}]},{"featureType":"all","elementType":"geometry.fill","stylers":[{"hue":"#ffbb00"}]},{"featureType":"all","elementType":"labels.text.fill","stylers":[{"hue":"#ffbb00"}]}]',
        	'Blue Water'       => '[{"featureType":"administrative","elementType":"labels.text.fill","stylers":[{"color":"#444444"}]},{"featureType":"landscape","elementType":"all","stylers":[{"color":"#f2f2f2"}]},{"featureType":"poi","elementType":"all","stylers":[{"visibility":"off"}]},{"featureType":"road","elementType":"all","stylers":[{"saturation":-100},{"lightness":45}]},{"featureType":"road.highway","elementType":"all","stylers":[{"visibility":"simplified"}]},{"featureType":"road.arterial","elementType":"labels.icon","stylers":[{"visibility":"off"}]},{"featureType":"transit","elementType":"all","stylers":[{"visibility":"off"}]},{"featureType":"water","elementType":"all","stylers":[{"color":"#46bcec"},{"visibility":"on"}]}]',
        	'Red Hat Antwerp'  => '[{"featureType":"administrative","elementType":"labels.text.fill","stylers":[{"color":"#444444"}]},{"featureType":"landscape","elementType":"all","stylers":[{"color":"#f2f2f2"}]},{"featureType":"poi","elementType":"all","stylers":[{"visibility":"off"}]},{"featureType":"poi.business","elementType":"geometry.fill","stylers":[{"visibility":"on"}]},{"featureType":"road","elementType":"all","stylers":[{"saturation":-100},{"lightness":45}]},{"featureType":"road.highway","elementType":"all","stylers":[{"visibility":"simplified"}]},{"featureType":"road.arterial","elementType":"labels.icon","stylers":[{"visibility":"off"}]},{"featureType":"transit","elementType":"all","stylers":[{"visibility":"off"}]},{"featureType":"water","elementType":"all","stylers":[{"color":"#b4d4e1"},{"visibility":"on"}]}]',
        	'Subtle Grayscale' => '[{"featureType":"landscape","stylers":[{"saturation":-100},{"lightness":65},{"visibility":"on"}]},{"featureType":"poi","stylers":[{"saturation":-100},{"lightness":51},{"visibility":"simplified"}]},{"featureType":"road.highway","stylers":[{"saturation":-100},{"visibility":"simplified"}]},{"featureType":"road.arterial","stylers":[{"saturation":-100},{"lightness":30},{"visibility":"on"}]},{"featureType":"road.local","stylers":[{"saturation":-100},{"lightness":40},{"visibility":"on"}]},{"featureType":"transit","stylers":[{"saturation":-100},{"visibility":"simplified"}]},{"featureType":"administrative.province","stylers":[{"visibility":"off"}]},{"featureType":"water","elementType":"labels","stylers":[{"visibility":"on"},{"lightness":-25},{"saturation":-100}]},{"featureType":"water","elementType":"geometry","stylers":[{"hue":"#ffff00"},{"lightness":-25},{"saturation":-97}]}]'
		);

		/**
		 * Front-end display of widget.
		 *
		 * @see WP_Widget::widget()
		 *
		 * @param array $args Widget arguments.
		 * @param array $instance Saved values from database.
		 */
		public function widget( $args, $instance ) {

			// Enqueue Google Maps API script
			wp_enqueue_script( 'google-maps' );

			extract( $args );
			extract( $instance );
			echo $before_widget;

			// Drag mobile set to true or false based on widget option
			$mobile_drag = empty( $instance['mobile_drag'] ) ? 0 : 1;

			if ( get_theme_mod( 'qt_custom_google_api' ) != '' ) { ?>
				<div 
					class="qt-map"
					data-lat="<?php echo esc_attr( $lat ); ?>"
					data-lng="<?php echo esc_attr( $lng ); ?>"
					<?php if( $instance['title'] ) : ?>
						data-title="<?php echo esc_attr( $title ); ?>"
					<?php endif; ?>
					data-zoom="<?php echo absint( $zoom ); ?>"
					data-type="<?php echo esc_attr( $type ); ?>"
					data-style="<?php echo esc_attr( $this->map_styles[$style] ); ?>"
					<?php if( $instance['pin'] ) : ?>
						data-pin="<?php echo esc_url( $pin ); ?>"
					<?php endif; ?>
					data-dragmobile="<?php echo esc_attr( $mobile_drag ); ?>"
					style="height: <?php echo absint( $height ); ?>px;"
				></div>
			<?php } else { ?>
				<div class="map-alert text-center">
					<?php echo esc_html_e( 'Please add your Google Maps API key at Appearance -> Customize -> Theme Options -> Custom -> Google Maps API key. Please do not forget to setup your (billing) account at Google', 'the-landscaper-wp' ); ?>
				</div>
			<?php } ?>

			<?php
			echo $after_widget;
		}

		/**
		 * Sanitize widget form values as they are saved.
		 *
		 * @see WP_Widget::update()
		 *
		 * @param array $new_instance Values just sent to be saved.
		 * @param array $old_instance Previously saved values from database.
		 *
		 * @return array Updated safe values to be saved.
		 */
		public function update( $new_instance, $old_instance ) {
			$instance = array();

			$instance['lat']    	 = sanitize_text_field( $new_instance['lat'] );
			$instance['lng']    	 = sanitize_text_field( $new_instance['lng'] );
			$instance['title']  	 = wp_kses_post( $new_instance['title'] );
			$instance['zoom']   	 = absint( $new_instance['zoom'] );
			$instance['type']   	 = sanitize_key( $new_instance['type'] );
			$instance['style']  	 = sanitize_text_field( $new_instance['style'] );
			$instance['pin']    	 = esc_url( $new_instance['pin'] );
			$instance['height'] 	 = absint( $new_instance['height'] );
			$instance['mobile_drag'] = ! empty( $new_instance['mobile_drag'] ) ? sanitize_key( $new_instance['mobile_drag'] ) : '';

			return $instance;
		}

		/**
		 * Back-end widget form.
		 *
		 * @see WP_Widget::form()
		 *
		 * @param array $instance Previously saved values from database.
		 */
		public function form( $instance ) {
			$lat    	 = isset( $instance['lat'] ) ? $instance['lat'] : '40.724545';
			$lng    	 = isset( $instance['lng'] ) ? $instance['lng'] : '-73.941860';
			$title  	 = isset( $instance['title'] ) ? $instance['title'] : '';
			$zoom   	 = isset( $instance['zoom'] ) ? $instance['zoom'] : 12;
			$type   	 = isset( $instance['type'] ) ? $instance['type'] : 'roadmap';
			$style  	 = isset( $instance['style'] ) ? $instance['style'] : 'Subtle Grayscale';
			$pin    	 = isset( $instance['pin'] ) ? $instance['pin'] : '';
			$height 	 = isset( $instance['height'] ) ? $instance['height'] : 400;
			$mobile_drag = empty( $instance['mobile_drag'] ) ? '' : $instance['mobile_drag'];
			?>

			<div class="qt-widget-notice">
					<p>
						<i class="dashicons dashicons-info"></i>
						<strong>
							<?php printf( esc_html__( "The API embedded map requirements has been changed by Google. You need to setup a Google billing account and create an API key for your map. Please see %s for more information.", 'the-landscaper-wp' ), '<a href="'. esc_url( '//cloud.google.com/maps-platform/pricing/' ) .'" target="_blank">this page</a>' ); ?>
							<br><br>
							<?php
								esc_html_e( 'To use the free version of Google Maps please select the "Google Maps Free" widget from the available widgets', 'the-landscaper-wp' );
							?>
						</strong>
					</p>
				</div>

			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'lat' ) ); ?>"><?php esc_html_e( 'Latitude of the location', 'the-landscaper-wp' ); ?> </label> <em><?php printf( esc_html__( "(You can find the latitude and longitude from %s)", 'the-landscaper-wp'  ), '<a href="'. esc_url( 'http://www.latlong.net/' ) .'" target="_blank">this website</a>' ); ?></em> <br>
				<input type="text" id="<?php echo esc_attr( $this->get_field_id( 'lat' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'lat' ) ); ?>" value="<?php echo esc_attr( $lat ); ?>" />
			</p>

			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'lng' ) ); ?>"><?php esc_html_e( 'Longitude of the location:', 'the-landscaper-wp' ); ?></label><br>
				<input type="text" id="<?php echo esc_attr( $this->get_field_id( 'lng' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'lng' ) ); ?>" value="<?php echo esc_attr( $lng ); ?>" />
			</p>

			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>"><?php esc_html_e( 'Infowindow content:', 'the-landscaper-wp' ); ?></label><br>
				<input class="widefat" type="text" id="<?php echo esc_attr( $this->get_field_id( 'title' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'title' ) ); ?>" value="<?php echo esc_attr( $title ); ?>" />
			</p>

			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'zoom' ) ); ?>"><?php esc_html_e( 'Map Zoom:', 'the-landscaper-wp' ); ?></label><br>
				<select id="<?php echo esc_attr( $this->get_field_id( 'zoom' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'zoom' ) ); ?>">
				<?php for ( $i=1; $i < 25; $i++ ) : ?>
					<option value="<?php echo esc_attr( $i ); ?>" <?php selected( $zoom, $i ); ?>><?php echo esc_html( $i ); ?></option>
				<?php endfor; ?>
				</select>
			</p>

			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'type' ) ); ?>"><?php esc_html_e( 'Map Type:', 'the-landscaper-wp' ); ?></label><br>
				<select id="<?php echo esc_attr( $this->get_field_id( 'type' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'type' ) ); ?>">
					<option value="roadmap" <?php selected( $type, 'roadmap' ); ?>>Roadmap</option>
					<option value="satellite" <?php selected( $type, 'satellite' ); ?>>Satellite</option>
					<option value="hybrid" <?php selected( $type, 'hybrid' ); ?>>Hybrid</option>
					<option value="terrain" <?php selected( $type, 'terrain' ); ?>>Terrain</option>
				</select>
			</p>

			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'style' ) ); ?>"><?php esc_html_e( 'Map Style:', 'the-landscaper-wp'  ); ?></label><br>
				<select id="<?php echo esc_attr( $this->get_field_id( 'style' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'style' ) ); ?>">
				<?php foreach ( $this->map_styles as $style_name => $map_style ) : ?>
					<option value="<?php echo esc_attr( $style_name ); ?>" <?php selected( $style, $style_name ); ?>><?php echo esc_html( $style_name ); ?></option>
				<?php endforeach; ?>
				</select>
			</p>

			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'height' ) ); ?>"><?php esc_html_e( 'Height of map (e,g. 400)', 'the-landscaper-wp'  ); ?></label><br>
				<input id="<?php echo esc_attr( $this->get_field_id( 'height' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'height' ) ); ?>" type="number" value="<?php echo esc_attr( $height ); ?>" />
			</p>

			<p>
				<label for="<?php echo esc_attr( $this->get_field_id( 'pin' ) ); ?>"><?php esc_html_e( 'Custom Marker URL (Leave empty for default marker)', 'the-landscaper-wp'  ); ?></label>
				<input class="widefat" type="text" id="<?php echo esc_attr( $this->get_field_id( 'pin' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'pin' ) ); ?>" value="<?php echo esc_url( $pin ); ?>" />
			</p>

			<p>
				<input class="checkbox" type="checkbox" <?php checked( $mobile_drag, 'on' ); ?> id="<?php echo esc_attr( $this->get_field_id( 'mobile_drag' ) ); ?>" name="<?php echo esc_attr( $this->get_field_name( 'mobile_drag' ) ); ?>" value="on" />
				<label for="<?php echo esc_attr( $this->get_field_id( 'mobile_drag' ) ); ?>"><?php esc_html_e('Disable map scrolling on mobile?', 'the-landscaper-wp' ); ?></label>
			</p>

			<?php
		}
	}
}